; ================================================================
; ATARI 800XL SOKOBAN (6502) - XASM
; Wersja: v26 (FINAL RELEASE - Full Comments & Polish)
; Autorzy: Gemini 3 Pro & Nosty
; ================================================================

; --- REJESTRY SPRZETOWE (HARDWARE) ---
DMACTL  equ $D400   ; Kontrola DMA (Direct Memory Access)
PORTA   equ $D300   ; Port sterowania (Joystick)
TRIG0   equ $D010   ; Przycisk joysticka (0 = wcisniety)
NMIEN   equ $D40E   ; Wlaczenie przerwan NMI (VBI)
WSYNC   equ $D40A   ; Oczekiwanie na pozioma synchronizacje

; --- UKLAD DZWIEKOWY (POKEY) ---
AUDF0   equ $D200   ; Czestotliwosc kanalu 0
AUDC0   equ $D201   ; Kontrola (glosnosc/znieksztalcenie) kanalu 0
AUDF1   equ $D202   ; Kanal 1
AUDC1   equ $D203
AUDF2   equ $D204   ; Kanal 2
AUDC2   equ $D205
AUDF3   equ $D206   ; Kanal 3
AUDC3   equ $D207
AUDCTL  equ $D208   ; Ogolna kontrola audio
SKCTL   equ $D20F   ; Kontrola serial port / init POKEY

; --- REJESTRY SYSTEMOWE (SHADOW) ---
SDMCTL  equ $022F   ; Cien DMACTL
SDLSTL  equ $0230   ; Wskaznik do Display List
CHBAS   equ $02F4   ; Adres zestawu znakow (Fontu)
COLOR0  equ $02C4   ; Rejestr koloru 0
COLOR1  equ $02C5   ; Rejestr koloru 1
COLOR2  equ $02C6   ; Rejestr koloru 2
COLOR3  equ $02C7   ; Rejestr koloru 3
COLOR4  equ $02C8   ; Rejestr koloru 4 (Tlo)
RTCLOK  equ $14     ; Zegar systemowy (zwiekszany co ramke - 1/50s)

CH      equ $02FC   ; Ostatnio wcisniety klawisz
STICK0  equ $0278   ; Cien joysticka (nieuzywany, uzywamy PORTA)

; --- STALE ---
JOY_DELAY   equ 10      ; Opoznienie miedzy ruchami (plynnosc sterowania)
KEY_ESC     equ $1C     ; Kod klawisza ESC
KEY_SPACE   equ $21     ; Kod klawisza SPACJA
KEY_NONE    equ $FF     ; Brak wcisnietego klawisza

; Konfiguracja kolorow
TXT_COLOR   equ $46     ; Jasny czerwony (dla menu)
BG_COLOR    equ $00     ; Czarny (dla menu)

; --- MAPA PAMIECI ---
MAP_BUF     equ $5000   ; Bufor logiczny mapy (tu odbywa sie gra)
TXT_MEM     equ $5400   ; Pamiec ekranu tytulowego (ok. 1KB)
SCR_MEM     equ $6000   ; Pamiec ekranu gry (ok. 1KB)
FONT_NEW    equ $7000   ; Zmodyfikowany zestaw znakow (ok. 1KB)

; --- STRONA ZEROWA (Zmienne szybkie) ---
    org $80
ptr_src     org *+2     ; Wskaznik zrodlowy (16-bit)
ptr_dst     org *+2     ; Wskaznik docelowy (16-bit)
p_x         org *+1     ; Aktualna pozycja gracza X
p_y         org *+1     ; Aktualna pozycja gracza Y
next_x      org *+1     ; Planowana pozycja X
next_y      org *+1     ; Planowana pozycja Y
map_w       org *+1     ; Szerokosc biezacej mapy
map_h       org *+1     ; Wysokosc biezacej mapy
level_idx   org *+1     ; Numer aktualnego poziomu (0-13)
timer       org *+1     ; Licznik opoznienia joysticka
tmp         org *+1     ; Zmienna pomocnicza 1
tmp2        org *+1     ; Zmienna pomocnicza 2
move_dx     org *+1     ; Wektor ruchu X (-1, 0, 1)
move_dy     org *+1     ; Wektor ruchu Y (-1, 0, 1)
target_tile org *+1     ; Co znajduje sie na polu docelowym
behind_tile org *+1     ; Co znajduje sie za skrzynia

; Zmienne dzwiekowe (liczniki czasu trwania)
sfx_cnt0    org *+1     ; Kanal 0 (Kroki)
sfx_cnt1    org *+1     ; Kanal 1 (Pchanie)
sfx_cnt2    org *+1     ; Kanal 2 (Sukces)
sfx_block_step org *+1  ; Flaga blokady dzwieku kroku przy pchaniu

; ================================================================
; GLOWNY PROGRAM
; ================================================================
    org $2000

Start
    lda #0
    sta SDMCTL      ; Wylacz obraz podczas inicjalizacji
    sta NMIEN       ; Wylacz przerwania
    
    jsr InitColors  ; Ustaw kolory poczatkowe
    jsr InitFont    ; Przygotuj grafike w RAM
    jsr InitSound   ; Wlacz POKEY
    
    lda #$40
    sta NMIEN       ; Wlacz przerwania VBI (niezbedne dla RTCLOK)
    
    lda #0
    sta level_idx   ; Zacznij od poziomu 0 (pierwszego)

ShowTitle
    ; Konfiguracja Display List ekranu tytulowego
    lda <DL_TITLE
    sta SDLSTL
    lda >DL_TITLE
    sta SDLSTL+1
    
    jsr SetTitleColors ; Ustaw palete dla menu (czarno-czerwona)
    
    lda #$E0
    sta CHBAS       ; Uzyj systemowego fontu Atari (dla napisow)
    
    jsr ClearTextMem   ; Wyczysc pamiec ekranu tytulowego
    jsr PrintTitleText ; Wypisz teksty menu
    
    lda #$22        ; Wlacz DMA (obraz widoczny)
    sta SDMCTL

WaitTitle
    ; Oczekiwanie na nacisniecie FIRE
    jsr WaitFrame
    lda TRIG0
    beq WaitRelease ; Jesli trzymany, czekaj na puszczenie
    jmp CheckPress
WaitRelease
    lda TRIG0
    beq WaitRelease
CheckPress
    lda TRIG0
    bne CheckPress  ; Czekaj az zostanie wcisniety (0)
    
    ; --- START ROZGRYWKI ---
StartGame
    lda #0
    sta SDMCTL      ; Wylacz obraz na moment ladowania
    
    jsr LoadLevel   ; Wczytaj dane poziomu i narysuj mape
    jsr SetGameColors ; Ustaw palete dla gry (braz, niebieski, zielony)
    
    ; Przelacz Display List na ekran gry
    lda <DL_GAME
    sta SDLSTL
    lda >DL_GAME
    sta SDLSTL+1
    
    lda #>FONT_NEW
    sta CHBAS       ; Przelacz na nasz zmodyfikowany font ($70)
    
    lda #$22
    sta SDMCTL      ; Wlacz obraz
    
    lda #0
    sta timer       ; Reset timera sterowania
    lda #KEY_NONE
    sta CH          ; Wyczysc bufor klawiatury

GameLoop
    jsr WaitFrame   ; Synchronizacja (1 ramka = 1/50s)
    
    ; 1. Obsluga Klawiatury (ESC, SPACJA)
    lda CH
    cmp #KEY_NONE
    beq CheckJoy
    
    cmp #KEY_ESC
    bne TrySpace
    lda #KEY_NONE
    sta CH
    jmp StartGame   ; ESC -> Restart poziomu
    
TrySpace
    cmp #KEY_SPACE
    bne CheckJoy
    lda #KEY_NONE
    sta CH
    jmp LevelDone   ; SPACE -> Pomin poziom (Cheat)

    ; 2. Obsluga Joysticka
CheckJoy
    lda timer
    beq ReadStick   ; Jesli timer=0, czytaj joystick
    dec timer       ; Jesli timer>0, zmniejsz i czekaj
    jmp GameLoop

ReadStick
    ; Odczyt bezposredni z PORTA ($D300)
    lda PORTA
    and #15         ; Interesuja nas tylko dolne 4 bity (kierunki)
    cmp #15         ; 1111 = brak ruchu
    beq GameLoop
    
    ; Ruch wykryty - ustaw opoznienie
    tay             ; Zachowaj stan joya w Y
    lda #JOY_DELAY
    sta timer
    tya             ; Przywroc stan joya do A
    
    ldx #0
    stx move_dx
    stx move_dy
    
    ; Rozpoznawanie kierunkow (0 = wcisniety)
    lsr @           ; Bit 0 -> Carry (GORA)
    bcs CheckDown
    dec move_dy
    jmp DoMove
CheckDown
    lsr @           ; Bit 1 -> Carry (DOL)
    bcs CheckLeft
    inc move_dy
    jmp DoMove
CheckLeft
    lsr @           ; Bit 2 -> Carry (LEWO)
    bcs CheckRight
    dec move_dx
    jmp DoMove
CheckRight
    lsr @           ; Bit 3 -> Carry (PRAWO)
    bcs GameLoop
    inc move_dx
    
DoMove
    jsr TryMoveLogic ; Wykonaj logike ruchu
    jsr CheckWin     ; Sprawdz czy wygralismy
    beq LevelDone    ; Jesli Z=1 (Wygrana), skocz dalej
    jmp GameLoop

LevelDone
    ; Maly delay po wygranej
    ldx #30
WaitLvl
    jsr WaitFrame
    dex
    bne WaitLvl

    inc level_idx   ; Nastepny poziom
    lda level_idx
    cmp #14         ; Czy to byl ostatni poziom (mamy 14)?
    beq AllDone
    jmp StartGame   ; Laduj kolejny

AllDone
    jmp ShowEndScreen

; ================================================================
; LOGIKA GRY
; ================================================================
TryMoveLogic
    ; Reset flagi dzwieku kroku
    lda #0
    sta sfx_block_step

    ; Oblicz wspolrzedne docelowe (next_x, next_y)
    lda p_x
    clc
    adc move_dx
    sta next_x
    lda p_y
    clc
    adc move_dy
    sta next_y
    
    ; Sprawdz co jest na polu docelowym
    ldx next_x
    ldy next_y
    jsr GetMapTile
    sta target_tile
    
    ; Decyzja w zaleznosci od obiektu
    cmp #9          ; Tlo (pustka) -> Blokada
    beq MoveBlock
    cmp #4          ; Sciana -> Blokada
    beq MoveBlock
    cmp #3          ; Skrzynia -> Proba pchniecia
    beq TryPush
    cmp #2          ; Skrzynia na slocie -> Proba pchniecia
    beq TryPush
    
    ; Wolne pole (Podloga/Slot) -> Ruch
    ldx next_x
    ldy next_y
    jsr MovePlayer
    rts

TryPush
    ; PCHAMY! Blokujemy dzwiek kroku, gramy dzwiek pchania
    lda #1
    sta sfx_block_step
    jsr PlayPush

    ; Oblicz wspolrzedne pola ZA skrzynia
    lda next_x
    clc
    adc move_dx
    sta tmp
    lda next_y
    clc
    adc move_dy
    sta tmp2
    
    ; Sprawdz co jest za skrzynia
    ldx tmp
    ldy tmp2
    jsr GetMapTile
    sta behind_tile
    
    ; Czy mozna tam wepchnac skrzynie?
    cmp #9          ; Tlo -> Nie
    beq MoveBlock
    cmp #4          ; Sciana -> Nie
    beq MoveBlock
    cmp #3          ; Druga skrzynia -> Nie
    beq MoveBlock
    cmp #2          ; Skrzynia na slocie -> Nie
    beq MoveBlock
    
    ; OK, pchamy!
    ldx tmp
    ldy tmp2
    jsr CalcMapAddr ; Oblicz adres w mapie dla nowej pozycji skrzyni
    
    ldy #0
    lda behind_tile
    cmp #1          ; Czy wpychamy na slot?
    beq BoxToSlot
    lda #3          ; Nie, zwykla skrzynia
    jmp SetBox
BoxToSlot
    ; Skrzynia weszla na slot!
    jsr PlaySuccess ; Dzwiek sukcesu
    lda #2          ; Typ: Skrzynia na slocie
    
SetBox
    sta (ptr_dst),y ; Zapisz skrzynie w nowym miejscu mapy
    
    ; Narysuj skrzynie w nowym miejscu na ekranie
    ldx tmp
    ldy tmp2
    jsr DrawTileAt
    
    ; Przesun gracza na miejsce, gdzie stala skrzynia
    ldx next_x
    ldy next_y
    jsr MovePlayer
    rts

MoveBlock
    rts             ; Ruch zablokowany, nic nie rob

MovePlayer
    ; Procedura przesuwa gracza z (p_x, p_y) na (next_x, next_y)
    stx next_x
    sty next_y
    
    ; 1. "Wyczysc" stara pozycje gracza (przywroc podloge/slot)
    jsr CalcPlayerAddr
    ldy #0
    lda (ptr_dst),y
    cmp #5          ; Czy stal na slocie?
    beq WasOnSlot
    lda #0          ; Nie, stal na podlodze
    jmp RestoreOld
WasOnSlot
    lda #1          ; Tak, zostaw slot
RestoreOld
    sta (ptr_dst),y
    
    ; Odswiez graficznie stare pole
    pha
    ldx p_x
    ldy p_y
    pla
    jsr DrawTileAt
    
    ; 2. Postaw gracza na nowej pozycji w mapie
    ldx next_x
    ldy next_y
    jsr CalcMapAddr
    ldy #0
    lda (ptr_dst),y
    
    cmp #3          ; Jesli weszlismy na skrzynie (ktora wlasnie pchnelismy), to pod spodem jest podloga
    beq IsFloor
    cmp #2
    beq IsSlot
    cmp #1          ; Wchodzimy na slot
    beq IsSlot
IsFloor
    lda #6          ; Gracz na podlodze
    bne PutPlayer
IsSlot
    lda #5          ; Gracz na slocie
PutPlayer
    sta (ptr_dst),y
    
    ; Zapisz nowa pozycje globalna (zabezpieczajac A)
    pha
    lda next_x
    sta p_x
    lda next_y
    sta p_y
    pla
    
    ; Narysuj gracza w nowym miejscu
    ldx p_x
    ldy p_y
    jsr DrawTileAt
    
    ; Odtworz dzwiek kroku (jesli nie byl zablokowany przez pchanie)
    lda sfx_block_step
    bne NoStepSnd
    jsr PlayStep
NoStepSnd
    rts

; ================================================================
; SYSTEM DZWIEKOWY (POKEY)
; ================================================================
InitSound
    lda #3
    sta SKCTL       ; Reset i inicjalizacja POKEY
    lda #0
    sta AUDCTL
    sta AUDC0
    sta AUDC1
    sta AUDC2
    sta AUDC3
    sta sfx_cnt0
    sta sfx_cnt1
    sta sfx_cnt2
    rts

UpdateSound
    ; Procedura wywolywana co ramke (VBI)
    
    ; --- Kanal 0: Kroki (Szum) ---
    lda sfx_cnt0
    beq UpdSnd1
    dec sfx_cnt0
    ora #$80        ; Distortion 8 (Szum bialy)
    sta AUDC0
    lda #$20        ; Pitch
    sta AUDF0
    jmp UpdSnd2
UpdSnd1
    lda #0
    sta AUDC0       ; Wycisz

UpdSnd2
    ; --- Kanal 1: Pchanie (Niski szum) ---
    lda sfx_cnt1
    beq UpdSnd3
    dec sfx_cnt1
    lsr @           ; Zmniejsz glosnosc (Timer / 2)
    ora #$00        ; Distortion 0 (Niski, brudny ton)
    sta AUDC1
    lda #$60        ; Pitch
    sta AUDF1
    jmp UpdSnd4
UpdSnd3
    lda #0
    sta AUDC1

UpdSnd4
    ; --- Kanal 2: Sukces (Czysty ton) ---
    lda sfx_cnt2
    beq UpdSndEnd
    dec sfx_cnt2
    lsr @           ; Glosnosc
    ora #$A0        ; Distortion A (Czysty ton)
    sta AUDC2
    lda #$1D        ; Nuta
    sta AUDF2
    jmp UpdSndRet
UpdSndEnd
    lda #0
    sta AUDC2
UpdSndRet
    rts

PlayStep
    lda #4          ; Czas trwania kroku
    sta sfx_cnt0
    rts

PlayPush
    lda #16         ; Czas trwania pchania
    sta sfx_cnt1
    rts

PlaySuccess
    lda #30         ; Czas trwania dzwieku sukcesu
    sta sfx_cnt2
    rts

; ================================================================
; PROCEDURY POMOCNICZE
; ================================================================
WaitFrame
    lda RTCLOK
    sta tmp
WaitLoop
    lda RTCLOK
    cmp tmp
    beq WaitLoop
    
    jsr UpdateSound ; Aktualizacja dzwieku w tle
    rts

CalcMapAddr
    ; Oblicza: MAP_BUF + Y * map_w + X
    ; Wynik w ptr_dst
    lda #<MAP_BUF
    sta ptr_dst
    lda >MAP_BUF
    sta ptr_dst+1
    tya
    beq AddX
CalcLoop
    lda ptr_dst
    clc
    adc map_w
    sta ptr_dst
    bcc NoCarryMap
    inc ptr_dst+1
NoCarryMap
    dey
    bne CalcLoop
AddX
    txa
    clc
    adc ptr_dst
    sta ptr_dst
    bcc RetMap
    inc ptr_dst+1
RetMap
    rts

CalcPlayerAddr
    ldx p_x
    ldy p_y
    jmp CalcMapAddr

GetMapTile
    jsr CalcMapAddr
    ldy #0
    lda (ptr_dst),y
    rts

DrawTileAt
    ; Rysuje element A na pozycji X, Y
    stx tmp
    sty tmp2
    pha             ; Zachowaj typ elementu
    
    ; Oblicz adres ekranu: SCR_MEM + Y*40 + X*2
    lda #<SCR_MEM
    sta ptr_src
    lda >SCR_MEM
    sta ptr_src+1
    
    ldy tmp2
    beq CalcScrX
ScrLoop
    lda ptr_src
    clc
    adc #40
    sta ptr_src
    bcc NoCarryScr
    inc ptr_src+1
NoCarryScr
    dey
    bne ScrLoop
    
CalcScrX
    lda tmp
    asl @           ; X * 2 (bo 2 znaki na kafel)
    clc
    adc ptr_src
    sta ptr_src
    bcc PtrReady
    inc ptr_src+1
PtrReady
    
    pla             ; Przywroc typ
    
    ; Mapowanie specjalne
    cmp #0          ; Podloga
    beq IsLog0
    cmp #9          ; Tlo
    beq IsLog9
    
    ; Sprawdz czy to gracz (zeby zmienic kolor)
    cmp #5
    beq IsPlayer
    cmp #6
    beq IsPlayer
    
    ; Standardowy element: Znak = Typ * 2
    asl @
    tax
    jmp DoDraw

IsPlayer
    ; Gracz: Znak = Typ * 2 + 128 (bit 7) -> 5. kolor (COLOR3)
    asl @
    ora #$80
    tax
    jmp DoDraw

IsLog0
    ldx #14         ; Znak podlogi
    jmp DoDraw
IsLog9
    ldx #0          ; Znak tla (pusty)
    
DoDraw
    txa
    ldy #0
    sta (ptr_src),y ; Lewy znak
    inx
    iny
    txa
    sta (ptr_src),y ; Prawy znak
    rts

CheckWin
    ; Sprawdza czy na mapie zostaly puste sloty (1)
    lda #<MAP_BUF
    sta ptr_src
    lda >MAP_BUF
    sta ptr_src+1
    lda map_h
    sta tmp2
CheckLoopY
    lda map_w
    sta tmp
    ldy #0
CheckLoopX
    lda (ptr_src),y
    cmp #1          ; Pusty slot?
    beq NotWin
    cmp #5          ; Gracz na slocie (bez skrzyni)?
    beq NotWin
    inc ptr_src
    bne NoIncH
    inc ptr_src+1
NoIncH
    dec tmp
    bne CheckLoopX
    dec tmp2
    bne CheckLoopY
    lda #0          ; Wygrana (Z=1)
    rts
NotWin
    lda #1          ; Gramy dalej (Z=0)
    rts

LoadLevel
    ; Wczytuje poziom o numerze level_idx do bufora i rysuje go
    lda #<LevelsData
    sta ptr_src
    lda >LevelsData
    sta ptr_src+1
    
    ldx level_idx
    beq LevelFound
    
SkipLevels
    ldy #0
    lda (ptr_src),y ; W
    sta tmp
    iny
    lda (ptr_src),y ; H
    sta tmp2
    
    txa
    pha
    jsr MultiplyTmp ; Oblicz rozmiar (W*H)
    
    ; Przesun wskanik o rozmiar + 2 (naglowek)
    clc
    adc ptr_src
    sta ptr_src
    bcc Sk1
    inc ptr_src+1
Sk1
    lda ptr_src
    clc
    adc #2
    sta ptr_src
    bcc Sk2
    inc ptr_src+1
Sk2
    pla
    tax
    dex
    bne SkipLevels

LevelFound
    ldy #0
    lda (ptr_src),y
    sta map_w
    iny
    lda (ptr_src),y
    sta map_h
    
    lda ptr_src
    clc
    adc #2
    sta ptr_src
    bcc StartCopy
    inc ptr_src+1
StartCopy
    lda #<MAP_BUF
    sta ptr_dst
    lda >MAP_BUF
    sta ptr_dst+1
    
    lda map_h
    sta tmp2
LoadLoopY
    lda map_w
    sta tmp
LoadLoopX
    ldy #0
    lda (ptr_src),y
    
    ; Szukaj startowej pozycji gracza
    cmp #6
    beq FoundPlayer
    cmp #5
    beq FoundPlayer
    jmp StoreMap
FoundPlayer
    pha
    lda map_w
    sec
    sbc tmp
    sta p_x
    lda map_h
    sec
    sbc tmp2
    sta p_y
    pla
StoreMap
    ldy #0
    sta (ptr_dst),y
    inc ptr_src
    bne inc_dst
    inc ptr_src+1
inc_dst
    inc ptr_dst
    bne dec_cnt
    inc ptr_dst+1
dec_cnt
    dec tmp
    bne LoadLoopX
    dec tmp2
    bne LoadLoopY
    jsr DrawFullMap
    rts

MultiplyTmp
    lda #0
    ldx tmp2
MultLoop
    beq MultEnd
    clc
    adc tmp
    dex
    jmp MultLoop
MultEnd
    rts

DrawFullMap
    ; Czysci ekran i rysuje cala mape z bufora
    lda #<SCR_MEM
    sta ptr_dst
    lda >SCR_MEM
    sta ptr_dst+1
    ldx #4
    lda #0
ClearLoop
    ldy #0
ClLoop2
    sta (ptr_dst),y
    iny
    bne ClLoop2
    inc ptr_dst+1
    dex
    bne ClearLoop

    lda #0
    sta tmp2
DrMapY
    lda tmp2
    cmp map_h
    beq DrMapEnd
    lda #0
    sta tmp
DrMapX
    lda tmp
    cmp map_w
    beq DrMapNextLine
    ldx tmp
    ldy tmp2
    jsr GetMapTile
    ldx tmp
    ldy tmp2
    jsr DrawTileAt
    inc tmp
    jmp DrMapX
DrMapNextLine
    inc tmp2
    jmp DrMapY
DrMapEnd
    rts

ShowEndScreen
    lda <DL_TITLE
    sta SDLSTL
    lda >DL_TITLE
    sta SDLSTL+1
    
    jsr SetTitleColors
    
    lda #$E0
    sta CHBAS
    jsr ClearTextMem
    jsr PrintWinText
WaitEnd
    jsr WaitFrame
    lda TRIG0
    beq RestartAll
    jmp WaitEnd
RestartAll
    jmp Start

ClearTextMem
    ldx #0
    lda #0
clrtxt
    sta TXT_MEM,x
    sta TXT_MEM+$100,x
    sta TXT_MEM+$200,x
    sta TXT_MEM+$300,x
    inx
    bne clrtxt
    rts

; ================================================================
; TEKSTY
; ================================================================
PrintTitleText
    ldx #0
pt_loop1
    lda TxtTitle,x
    cmp #$FF
    beq pt_credits
    sta TXT_MEM+95,x
    inx
    jmp pt_loop1
pt_credits
    ldx #0
pt_loopC
    lda TxtCredits,x
    cmp #$FF
    beq pt_desc
    sta TXT_MEM+165,x
    inx
    jmp pt_loopC
pt_desc
    ldx #0
pt_loop2
    lda TxtDesc,x
    cmp #$FF
    beq pt_esc
    sta TXT_MEM+289,x
    inx
    jmp pt_loop2
pt_esc
    ldx #0
pt_loop3
    lda TxtEsc,x
    cmp #$FF
    beq pt_spc
    sta TXT_MEM+449,x
    inx
    jmp pt_loop3
pt_spc
    ldx #0
pt_loop4
    lda TxtSpc,x
    cmp #$FF
    beq pt_fire
    sta TXT_MEM+531,x
    inx
    jmp pt_loop4
pt_fire
    ldx #0
pt_loop5
    lda TxtFireStart,x
    cmp #$FF
    beq pt_ret
    sta TXT_MEM+810,x
    inx
    jmp pt_loop5
pt_ret
    rts

PrintWinText
    ldx #0
pw_loop1
    lda TxtCongrat,x
    cmp #$FF
    beq pw_msg
    sta TXT_MEM+412,x
    inx
    jmp pw_loop1
pw_msg
    ldx #0
pw_loop2
    lda TxtBeat,x
    cmp #$FF
    beq pw_fire
    sta TXT_MEM+492,x
    inx
    jmp pw_loop2
pw_fire
    ldx #0
pw_loop3
    lda TxtFire,x
    cmp #$FF
    beq pw_ret
    sta TXT_MEM+815,x
    inx
    jmp pw_loop3
pw_ret
    rts

; ================================================================
; KOLORY
; ================================================================
SetTitleColors
    lda #TXT_COLOR
    sta COLOR1
    lda #BG_COLOR
    sta COLOR2
    sta COLOR4
    rts

SetGameColors
    lda #$24        ; COLOR0: Braz (Sciany)
    sta COLOR0
    lda #$C2        ; COLOR1: Ciemna zielen (Podloga)
    sta COLOR1
    lda #$86        ; COLOR2: Niebieski (Skrzynie)
    sta COLOR2
    lda #$28        ; COLOR3: Pomaranczowy (Gracz)
    sta COLOR3
    lda #$00        ; COLOR4: Czarne tlo
    sta COLOR4
    rts

; ================================================================
; INICJALIZACJA FONTOW
; ================================================================
InitFont
    lda #>FONT_NEW
    sta ptr_dst+1
    lda #0
    sta ptr_dst
    ldx #4
cf_loop
    ldy #0
    lda #0
cf_inner
    sta (ptr_dst),y
    iny
    bne cf_inner
    inc ptr_dst+1
    dex
    bne cf_loop
    jsr SetCharSet
    rts

SetCharSet
    lda #<CharData
    sta ptr_src
    lda #>CharData
    sta ptr_src+1
    lda #<FONT_NEW+$10
    sta ptr_dst
    lda #>FONT_NEW
    sta ptr_dst+1
    ldy #0
CopyLoop
    lda (ptr_src),y
    sta (ptr_dst),y
    iny
    cpy #112
    bne CopyLoop
    rts

InitColors
    lda #0
    sta COLOR0
    sta COLOR1
    sta COLOR2
    sta COLOR3
    sta COLOR4
    rts

; ================================================================
; DANE (DISPLAY LISTY, GRAFIKA, TEKSTY, POZIOMY)
; ================================================================

    org $6400

DL_GAME
    dta $70,$70,$70
    dta $45,a(SCR_MEM) ; Antic 5 (Wysoki tekst, 4 kolory)
    dta 5,5,5,5,5,5,5,5,5,5,5,5
    dta 5,5
    dta $41,a(DL_GAME)

DL_TITLE
    dta $70,$70,$70
    dta $42,a(TXT_MEM) ; Antic 2 (Tryb tekstowy podstawowy)
    dta 2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2
    dta $41,a(DL_TITLE)

TxtTitle
    dta 33,41,0,51,47,43,47,34,33,46,$FF
TxtCredits
    dta 35,47,36,37,0,13,0,39,37,45,41,46,41,0,19,0,48,50,47,0,33,46,36,0,46,47,51,52,57,$FF
TxtDesc
    dta 48,53,51,40,0,34,47,56,37,51,0,41,46,52,47,0,51,44,47,52,51,$FF
TxtEsc
    dta 37,51,35,0,0,0,13,0,50,37,51,52,33,50,52,0,44,37,54,37,44,$FF
TxtSpc
    dta 51,48,33,35,37,0,13,0,51,43,41,48,0,44,37,54,37,44,$FF
TxtFireStart
    dta 48,50,37,51,51,0,38,41,50,37,0,52,47,0,51,52,33,50,52,$FF
TxtCongrat
    dta 35,47,46,39,50,33,52,53,44,33,52,41,47,46,51,1,$FF
TxtBeat
    dta 57,47,53,0,34,37,33,52,0,52,40,37,0,33,41,$FF
TxtFire
    dta 48,50,37,51,51,0,38,41,50,37,$FF

CharData
    dta $CC,$00,$C0,$00,$C0,$00,$C0,$CC ; Slot L
    dta $CF,$00,$03,$00,$03,$00,$03,$CF ; Slot R
    dta $FF,$F5,$DD,$D7,$D7,$DD,$F5,$FF ; BoxIn L
    dta $FF,$5F,$77,$D7,$D7,$77,$5F,$FF ; BoxIn R
    dta $FF,$F0,$CC,$C3,$C3,$CC,$F0,$FF ; Box L
    dta $FF,$0F,$33,$C3,$C3,$33,$0F,$FF ; Box R
    dta $00,$54,$54,$54,$00,$55,$55,$55 ; Sciana L
    dta $00,$55,$55,$55,$00,$54,$54,$54 ; Sciana R
    dta $0F,$3F,$F3,$FF,$FC,$3F,$0C,$FC ; Player L
    dta $F0,$FC,$CF,$FF,$3F,$FC,$30,$3F ; Player R
    dta $0F,$3F,$F3,$FF,$FC,$3F,$0C,$FC ; Player L (Copy)
    dta $F0,$FC,$CF,$FF,$3F,$FC,$30,$3F ; Player R (Copy)
    dta $88,$22,$88,$22,$88,$22,$88,$22 ; Floor L
    dta $88,$22,$88,$22,$88,$22,$88,$22 ; Floor R

LevelsData
    ; 1
    dta 11,11
    dta 9,9,9,4,4,4,9,9,9,9,9
    dta 9,9,4,4,0,4,9,4,4,4,4
    dta 9,4,4,0,0,4,4,4,0,0,4
    dta 4,4,0,3,0,0,0,0,0,0,4
    dta 4,0,0,0,6,3,0,4,0,0,4
    dta 4,4,4,0,3,4,4,4,0,0,4
    dta 9,9,4,0,0,4,1,1,0,0,4
    dta 9,4,4,0,4,4,1,4,0,4,4
    dta 9,4,0,0,0,0,0,0,4,4,9
    dta 9,4,0,0,0,0,0,4,4,9,9
    dta 9,4,4,4,4,4,4,4,9,9,9
    ; 2
    dta 9,9
    dta 9,4,4,9,4,4,4,4,4
    dta 4,4,9,4,4,0,1,0,4
    dta 4,9,4,4,0,3,1,0,4
    dta 9,4,4,0,3,0,0,0,4
    dta 4,4,0,3,6,0,4,4,4
    dta 4,0,3,0,0,4,4,9,9
    dta 4,1,1,0,4,4,9,4,4
    dta 4,0,0,0,4,9,4,4,9
    dta 4,4,4,4,4,9,4,9,9
    ; 3
    dta 13,10
    dta 9,9,4,4,4,4,4,4,4,4,4,4,4
    dta 9,4,4,0,0,0,0,0,4,0,0,6,4
    dta 4,4,4,0,3,0,3,3,4,0,0,0,4
    dta 4,0,4,4,3,0,0,0,0,3,3,0,4
    dta 4,0,0,4,0,0,3,0,4,0,0,0,4
    dta 4,4,4,4,4,4,0,4,4,4,4,4,4
    dta 4,1,1,0,1,1,3,0,4,2,4,4,9
    dta 4,0,1,1,0,0,0,0,4,4,4,9,9
    dta 4,0,0,1,1,4,4,4,4,4,9,9,9
    dta 4,4,4,4,4,4,4,4,4,9,9,9,9
    ; 4
    dta 9,9
    dta 4,4,4,4,4,4,4,4,4
    dta 4,1,0,1,0,0,0,0,4
    dta 4,1,3,1,0,1,0,0,4
    dta 4,4,0,4,4,4,6,0,4
    dta 9,4,0,0,3,0,0,4,4
    dta 9,4,0,3,3,0,4,4,9
    dta 9,4,0,0,3,0,4,9,9
    dta 9,4,0,0,4,4,4,9,9
    dta 9,4,4,4,4,9,9,9,9
    ; 5
    dta 15,12
    dta 9,9,4,4,4,4,4,4,4,4,4,4,4,9,9
    dta 4,4,4,1,0,0,1,3,1,0,0,1,4,4,4
    dta 9,4,4,0,3,0,0,3,0,0,3,0,4,4,9
    dta 9,9,4,4,0,1,1,3,1,1,0,4,4,9,9
    dta 9,9,9,4,4,3,4,3,4,3,4,4,9,9,9
    dta 9,9,9,9,4,1,3,0,3,1,4,9,9,9,9
    dta 9,9,9,9,4,0,0,6,0,0,4,9,9,9,9
    dta 9,9,9,9,4,4,4,0,4,4,4,9,9,9,9
    dta 9,9,9,4,4,0,3,0,3,0,4,4,9,9,9
    dta 9,9,9,4,1,0,0,3,0,0,1,4,9,9,9
    dta 9,9,9,4,4,4,0,1,0,4,4,4,9,9,9
    dta 9,9,9,9,9,4,4,4,4,4,9,9,9,9,9
    ; 6
    dta 19,10
    dta 9,9,9,9,9,9,9,9,9,9,9,4,4,4,4,4,4,9,9
    dta 9,9,9,9,4,4,4,4,9,9,4,4,0,0,0,0,4,9,9
    dta 9,9,4,4,4,0,0,4,9,9,4,0,0,4,4,0,4,4,4
    dta 4,4,4,0,0,0,0,4,4,4,4,0,4,0,0,0,3,0,4
    dta 4,0,0,3,0,6,0,1,1,1,2,1,1,0,0,3,0,0,4
    dta 4,0,3,0,3,0,0,4,4,0,4,4,4,0,0,0,4,4,4
    dta 4,4,4,0,4,4,4,0,0,0,4,9,4,4,4,4,4,9,9
    dta 9,4,0,0,0,0,0,0,4,4,4,9,9,9,9,9,9,9,9
    dta 9,4,0,0,0,4,4,4,4,9,9,9,9,9,9,9,9,9,9
    dta 9,4,4,4,4,4,9,9,9,9,9,9,9,9,9,9,9,9,9
    ; 7
    dta 13,11
    dta 9,9,9,9,4,4,4,4,4,4,4,9,9
    dta 9,9,9,9,4,0,0,0,0,0,4,4,9
    dta 4,4,4,4,4,0,4,4,4,0,0,4,4
    dta 4,0,0,0,0,0,0,0,4,0,0,4,4
    dta 4,6,3,2,2,2,1,0,4,4,3,0,4
    dta 4,0,0,4,0,0,0,0,4,4,0,1,4
    dta 4,4,0,0,4,4,0,0,4,0,3,0,4
    dta 9,4,4,0,0,4,4,4,4,1,3,1,4
    dta 9,9,4,4,0,0,0,0,0,0,0,0,4
    dta 9,9,9,4,4,4,4,4,4,0,0,4,4
    dta 9,9,9,9,9,9,9,9,4,4,4,4,9
    ; 8
    dta 11,9
    dta 9,9,4,4,4,4,9,9,9,9,9
    dta 4,4,4,0,0,4,4,4,4,9,9
    dta 4,0,0,0,6,0,0,0,4,4,9
    dta 4,0,4,1,0,1,4,1,4,4,4
    dta 4,0,3,3,3,0,3,3,3,0,4
    dta 4,4,4,1,4,1,4,1,4,0,4
    dta 9,4,4,0,0,0,0,0,0,0,4
    dta 9,9,4,4,4,4,0,0,4,4,4
    dta 9,9,9,9,9,4,4,4,4,9,9
    ; 9
    dta 10,8
    dta 9,4,4,4,4,4,4,4,9,9
    dta 4,4,0,0,0,0,0,4,9,4
    dta 4,0,0,2,1,3,1,4,9,9
    dta 4,0,0,2,1,4,1,4,4,4
    dta 4,0,4,3,6,3,3,0,0,4
    dta 4,0,0,0,4,4,0,4,0,4
    dta 4,4,4,4,4,4,0,0,0,4
    dta 9,9,9,9,9,4,4,4,4,4
    ; 10
    dta 9,11
    dta 9,9,9,4,4,4,4,9,9
    dta 9,9,9,4,6,0,4,9,9
    dta 9,9,4,4,0,0,4,9,9
    dta 9,9,4,0,1,3,4,9,9
    dta 9,9,4,3,1,0,4,9,9
    dta 4,4,4,1,1,3,4,4,4
    dta 4,0,0,1,1,3,0,0,4
    dta 4,0,3,0,3,0,4,0,4
    dta 4,4,4,4,4,0,4,0,4
    dta 9,9,9,9,4,0,0,0,4
    dta 9,9,9,9,4,4,4,4,4
    ; 11
    dta 9,8
    dta 9,9,9,9,9,4,4,4,4
    dta 9,9,9,9,9,4,0,0,4
    dta 4,4,4,4,4,4,3,1,4
    dta 4,0,0,0,3,0,3,1,4
    dta 4,0,3,6,3,1,1,1,4
    dta 4,0,3,3,3,1,1,4,4
    dta 4,0,0,3,0,1,1,4,9
    dta 4,4,4,4,4,4,4,4,9
    ; 12
    dta 15,6
    dta 4,4,4,4,4,9,4,4,4,4,4,4,4,4,9
    dta 4,0,0,0,4,4,4,0,1,0,3,0,0,4,9
    dta 4,0,0,0,3,0,2,1,1,0,4,3,0,4,4
    dta 4,4,0,3,4,0,1,1,2,0,3,0,0,6,4
    dta 9,4,0,0,3,0,1,0,4,4,4,0,0,0,4
    dta 9,4,4,4,4,4,4,4,4,9,4,4,4,4,4
    ; 13
    dta 17,9
    dta 9,9,9,9,9,4,4,4,4,4,4,4,9,9,9,9,9
    dta 4,4,4,4,4,4,0,0,0,0,0,4,4,4,4,4,4
    dta 4,0,0,1,0,1,1,3,4,3,1,1,0,1,0,0,4
    dta 4,0,0,3,0,3,0,0,1,0,0,3,0,3,0,0,4
    dta 4,4,4,3,4,4,4,4,6,4,4,4,4,3,4,4,4
    dta 4,0,0,3,0,3,0,0,1,0,0,3,0,3,0,0,4
    dta 4,0,0,1,0,1,1,3,4,3,1,1,0,1,0,0,4
    dta 4,4,4,4,4,4,0,0,0,0,0,4,4,4,4,4,4
    dta 9,9,9,9,9,4,4,4,4,4,4,4,9,9,9,9,9
    ; 14
    dta 18,11
    dta 9,9,9,9,9,9,9,9,9,9,9,4,4,4,4,9,9,9
    dta 9,4,4,4,4,4,4,4,4,4,4,4,0,0,4,9,9,9
    dta 9,4,0,0,3,0,3,0,3,0,3,0,0,0,4,4,9,9
    dta 9,4,0,0,4,0,4,0,4,0,4,0,4,3,4,4,9,9
    dta 9,4,4,1,0,1,0,1,0,1,0,1,0,1,4,9,9,9
    dta 9,9,4,3,4,0,4,0,4,0,4,0,4,3,4,4,4,4
    dta 4,4,4,1,0,1,0,1,0,1,0,1,0,1,0,0,0,4
    dta 4,4,4,3,4,0,4,0,4,0,4,0,4,0,6,0,0,4
    dta 9,9,4,0,0,0,3,0,3,0,3,0,3,0,0,4,4,4
    dta 9,9,4,0,0,4,4,4,4,4,4,4,4,4,4,4,9,9
    dta 9,9,4,4,4,4,9,9,9,9,9,9,9,9,9,9,9,9